/*
 * linux/sound/soc/codecs/tlv320aic32x4_tiload_drv.c
 *
 *
 * Copyright (C) 2011 Texas Instruments, Inc.
 *
 *
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * History:
 *
 * Rev 0.1 	 TiLoad support         	Mistral         191-01-2011
 *
 *         	 Added char driver for TiLoad support,  
 * 
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/kdev_t.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <asm/io.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <sound/soc.h>
#include <sound/control.h>
#include <linux/time.h>		/* For timing computations */

#include "tlv320aic32x4.h"
#include "tlv320aic3xxx_tiload_drv.h"

/* enable debug prints in the driver */
//#define DEBUG
#undef DEBUG

#ifdef DEBUG
#define dprintk(x...) 	printk(x)
#else
#define dprintk(x...)
#endif

#ifdef AIC3xxx_TiLoad

/* Function prototypes */
#ifdef REG_DUMP_AIC3xxx
static void aic3xxx_dump_page(struct i2c_client *i2c, u8 page);
#endif

/* externs */
extern int aic32x4_change_page(struct snd_soc_codec *codec, u8 new_page);
extern int aic32x4_write(struct snd_soc_codec *codec, u16 reg, u8 value);
int aic3xxx_driver_init(struct snd_soc_codec *codec);

#ifdef REG_DUMP_AIC3xxx
/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_dump_page
 * Purpose  : Read and display one codec register page, for debugging purpose
 *----------------------------------------------------------------------------
 */
static void aic3xxx_dump_page(struct i2c_client *i2c, u8 page)
{
	int i;
	u8 data;
	u8 test_page_array[256];

	dprintk("TiLoad DRIVER : %s\n", __FUNCTION__);
	aic32x4_change_page(codec, page);

	data = 0x0;

	i2c_master_send(i2c, data, 1);
	i2c_master_recv(i2c, test_page_array, 128);

	dprintk("\n------- AIC3xxx PAGE %d DUMP --------\n", page);
	for (i = 0; i < 128; i++) {
		dprintk(" [ %d ] = 0x%x\n", i, test_page_array[i]);
	}
}
#endif

/************** Dynamic  AIC3xxx driver, TI LOAD support  ***************/

static struct cdev *aic3xxx_cdev;
static int aic3xxx_major = 0;	/* Dynamic allocation of Mjr No. */
static int aic3xxx_opened = 0;	/* Dynamic allocation of Mjr No. */
static struct snd_soc_codec *aic3xxx_codec;
static unsigned int magic_num = 0xE0 ;
/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_open
 *
 * Purpose  : open method for aic3xxx-tiload programming interface
 *----------------------------------------------------------------------------
 */
static int aic3xxx_open(struct inode *in, struct file *filp)
{
	dprintk("TiLoad DRIVER : %s\n", __FUNCTION__);
	if (aic3xxx_opened) {
		printk("%s device is already opened\n", "aic3xxx");
		printk("%s: only one instance of driver is allowed\n",
		       "aic3xxx");
		return -1;
	}
	aic3xxx_opened++;
	return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_release
 *
 * Purpose  : close method for aic3xxx_tiload programming interface
 *----------------------------------------------------------------------------
 */
static int aic3xxx_release(struct inode *in, struct file *filp)
{
	dprintk("TiLoad DRIVER : %s\n", __FUNCTION__);
	aic3xxx_opened--;
	return 0;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_read
 *
 * Purpose  : read method for aic3xxx tiload programming interface
 *----------------------------------------------------------------------------
 */
static ssize_t aic3xxx_read(struct file *file, char __user * buf,
			    size_t count, loff_t * offset)
{
	static char rd_data[256];
	char reg_addr;
	size_t size;
	struct i2c_client *i2c = aic3xxx_codec->control_data;

	dprintk("TiLoad DRIVER : %s\n", __FUNCTION__);
	if (count > 128) {
		printk("Max 256 bytes can be read\n");
		count = 128;
	}

	/* copy register address from user space  */
	size = copy_from_user(&reg_addr, buf, 1);
	if (size != 0) {
		printk("read: copy_from_user failure\n");
		return -1;
	}

	if (i2c_master_send(i2c, &reg_addr, 1) != 1) {
		dprintk("Can not write register address\n");
		return -1;
	}
	/* read the codec/minidsp registers */
	size = i2c_master_recv(i2c, rd_data, count);

	if (size != count) {
		printk("read %d registers from the codec\n", size);
	}

	if (copy_to_user(buf, rd_data, size) != 0) {
		dprintk("copy_to_user failed\n");
		return -1;
	}

	return size;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_write
 *
 * Purpose  : write method for aic3xxx tiload programming interface
 *----------------------------------------------------------------------------
 */
static ssize_t aic3xxx_write(struct file *file, const char __user * buf,
			     size_t count, loff_t * offset)
{
	static char wr_data[258];
	size_t size;
	struct i2c_client *i2c = aic3xxx_codec->control_data;

	/* copy buffer from user space  */
	size = copy_from_user(wr_data, buf, count);
	if (size != 0) {
		printk("copy_from_user failure %d\n", size);
		return -1;
	}

	if (wr_data[0] == 0) {
		aic32x4_change_page(aic3xxx_codec, wr_data[1]);
	}

	size = i2c_master_send(i2c, wr_data, count);
	return size;
}

/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_ioctl
 *
 * Purpose  : ioctl method for aic3xxx tiload programming interface
 *----------------------------------------------------------------------------
 */
static int aic3xxx_ioctl(struct inode *inode, struct file *filp,
			 unsigned int cmd, unsigned long arg)
{
	int ret;

	if (_IOC_TYPE(cmd) != AIC3xxx_IOC_MAGIC)
		return -ENOTTY;

	dprintk("TiLoad DRIVER : %s\n", __FUNCTION__);
	
	switch (cmd) {
	case AIC3xxx_IOMAGICNUM_GET:
		ret = copy_to_user((char*)arg, &magic_num, sizeof(int));
		break;
	case AIC3xxx_IOMAGICNUM_SET:
		ret = copy_from_user(&magic_num, (char*)arg, sizeof(int));
		break;
	}
	return 0;
}

/******* File operations structure for aic3xxx tiload programming *********/
static struct file_operations aic3xxx_fops = {
	.owner = THIS_MODULE,
	.open = aic3xxx_open,
	.release = aic3xxx_release,
	.read = aic3xxx_read,
	.write = aic3xxx_write,
	.ioctl = aic3xxx_ioctl,
};

/*
 *----------------------------------------------------------------------------
 * Function : aic3xxx_driver_init
 *
 * Purpose  : Registeer a char driver for dynamic aic3xxx tiload programming
 *----------------------------------------------------------------------------
 */
int aic3xxx_driver_init(struct snd_soc_codec *codec)
{
	int result;
	dev_t dev = MKDEV(aic3xxx_major, 0);

	aic3xxx_codec = codec;

	dprintk("allocating dynamic major number\n");

	result = alloc_chrdev_region(&dev, 0, 1, "aic3xxx");

	if (result < 0) {
		dprintk("cannot allocate major number %d\n", aic3xxx_major);
		return result;
	}

	aic3xxx_major = MAJOR(dev);
	dprintk("allocated Major Number: %d\n", aic3xxx_major);

	aic3xxx_cdev = cdev_alloc();
	cdev_init(aic3xxx_cdev, &aic3xxx_fops);
	aic3xxx_cdev->owner = THIS_MODULE;
	aic3xxx_cdev->ops = &aic3xxx_fops;

	if (cdev_add(aic3xxx_cdev, dev, 1) < 0) {
		dprintk("aic3xxx_driver: cdev_add failed \n");
		unregister_chrdev_region(dev, 1);
		aic3xxx_cdev = NULL;
		return 1;
	}
	printk("Registered aic3xxx TiLoad Driver, Major number: %d \n", aic3xxx_major);
	return 0;
}

#endif
